unit Px7Table;

(*

This is a TTable descendant with an added method to make use of the
new Pdox7 descending indices. With this type of index, you can
determine the sortorder for every single field in the index.

AddPx7Index works just like TTable.Addindex except for an additional
string parameter named DescFields:

procedure AddPx7Index(const Name,
                     Fields,
                     DescFields: string;
                     Options: TIndexOptions);

With ixDescending in Options DescFields holds the list of fields
that will actually be indexed descending, i.e. for a full
descending index it equals the Fields parameter.

The following example creates an index that is descending on the
1st and 3rd and ascending on the 2nd field:

Px7Table1.AddPx7Index('AscDescMix',
                      'Field1;Field2;Field3',
                      'Field1;Field3',
                      [ixDescending]);

------------------------------------------------------------------

As it turned out that TUTILITY.DLL v2.52 (the one that ships with
PfW7 for Win3.1x) not only drops indices when rebuilding a table
but resets the table level back to 5, I added a method to set the
table level. This call makes your Paradox table a level 7 table:

Px7Table1.SetLevel('7');

NOTE the table has to be opened exclusive for this call.

A GetLevel method was added for completeness...
------------------------------------------------------------------

Note on the internals of desc indices...:

The sortorder of a field in the index is determined by setting the
correspondent entry in IDXDesc.iUnUsed to 1 for descending or
to 0 for ascending. There is one entry in iUnUsed for every field
in IDXDesc.aiKeyFld (i.e. for every field in the key...). The arrays 
for the above example would look like this:

aiKeyFld = (1,2,3);
iUnUsed  = (1,0,1);

See the code for EncodePx7IndexDesc for more details.

Reinhard Kalinke
[100417,3504@compuserve com],
 (c) 5/17/96

*)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DB, DBTables, DBIProcs, DBITypes;

type
  TPx7Table = class(TTable)
  private
    procedure EncodePx7IndexDesc(var IndexDesc: IDXDesc;
      const Name, Fields, DescFields: string;
      Options: TIndexOptions);
    function IsPxTable: Boolean;
  protected
  public
    procedure AddPx7Index(const Name, Fields, DescFields: string;
      Options: TIndexOptions);
    function GetLevel: string;
    procedure SetLevel(ALevel: string);
  published
  end;

procedure Register;

implementation

procedure TPx7Table.AddPx7Index(const Name, Fields,
                                DescFields: string;
                                Options: TIndexOptions);
var
  STableName: DBITBLNAME;
  IndexDesc: IDXDesc;
begin
  if not IsPxTable then
    AddIndex(Name, Fields, Options) {or raise an exception}
  else
  begin
    FieldDefs.Update;
    EncodePx7IndexDesc(IndexDesc, Name, Fields, DescFields, Options);
    if Active then
    begin
      CheckBrowseMode;
      CursorPosChanged;
      Check(DbiAddIndex(DBHandle, Handle, nil, nil, IndexDesc, nil));
    end else
    begin
      SetDBFlag(dbfTable, True);
      try
        Check(DbiAddIndex(DBHandle, nil, AnsiToNative(DBLocale, TableName,
          STableName, SizeOf(STableName) - 1), szParadox,
          IndexDesc, nil));
      finally
        SetDBFlag(dbfTable, False);
      end;
    end;
    DataEvent(dePropertyChange, 0);
  end;
end;

procedure TPx7Table.EncodePx7IndexDesc(var IndexDesc: IDXDesc;
  const Name, Fields, DescFields: string; Options: TIndexOptions);
var
  iPos, jPos: Integer;
begin
  FillChar(IndexDesc, SizeOf(IndexDesc), 0);
  with IndexDesc do
  begin
    AnsiToNative(DBLocale, Name, szName, SizeOf(szName) - 1);
    bPrimary := ixPrimary in Options;
    bUnique := ixUnique in Options;
    bDescending := ixDescending in Options;
    bCaseInsensitive := ixCaseInsensitive in Options;
    bMaintained := True;
    iPos := 1;
    while (iPos <= Length(Fields)) and (iFldsInKey < 16) do
    begin
      jPos := iPos;
      aiKeyFld[iFldsInKey] :=
        FieldDefs.Find(ExtractFieldName(Fields,iPos)).FieldNo;
      {this is the one that makes a field descending:}
      if bDescending
      and (pos(ExtractFieldName(Fields,jPos),DescFields)<>0) then
        iUnUsed[iFldsInKey] := 1;
      Inc(iFldsInKey);
    end;
  end;
end;

function TPx7Table.IsPxTable: Boolean;
begin
  Result := (TableType = ttParadox) or
    (CompareText(ExtractFileExt(TableName), '.DB') = 0);
end;

function TPx7Table.GetLevel: string;
var TblProps: CURProps;
begin
  Check(DBIGetCursorProps(Handle,TblProps));
  Result := IntToStr(TblProps.iTblLevel);
end;

procedure TPx7Table.SetLevel(ALevel: string);
var hDB: hDBIdb;
    TblProps: CURProps;
    pTableDesc: pCRTblDesc;
    pOptFldDesc: pFLDDesc;
    szLevel: DBIName;
begin
  pTableDesc := nil;
  pOptFldDesc := nil;
  Check(DBIGetCursorProps(Handle,TblProps));
  if (TblProps.iTblLevel <> StrToInt(ALevel)) then
  try
    DisableControls;
    GetMem(pTableDesc,sizeOf(CRTblDesc));
    FillChar(pTableDesc^,sizeOf(CRTblDesc),0);
    GetMem(pOptFldDesc,sizeOf(FLDDesc));
    FillChar(pOptFldDesc^,sizeOf(FLDDesc),0);
    with pTableDesc^ do
    begin
      AnsiToNative(DBLocale,TableName,szTblName,255);
      StrPCopy(szTblType,TblProps.szTableType);
      bProtected := TblProps.bProtected;
      StrPCopy(pOptFldDesc^.szName,'LEVEL');
      pOptFldDesc^.iLen := length(ALevel)+1;
      pFldOptParams := pOptFldDesc;
      StrPCopy(szLevel,ALevel);
      pOptData := @szLevel;
      iOptParams := 1;
      hDB := DBHandle;
      Close;
      Check( DBIDoRestructure(hDB,        {DB handle}
                              1,          {no of tbls (has to be 1)}
                              pTableDesc, {table data desc.}
                              nil,        {pSaveAs}
                              nil,        {pKeyViol}
                              nil,        {pProblem}
                              False) );   {Analyze only}
    end;
  finally
    if pTableDesc <> nil then
      FreeMem(pTableDesc,sizeOf(CRTblDesc));
    if pOptFldDesc <> nil then
      FreeMem(pOptFldDesc,SizeOf(FLDDesc));
    Open;
    EnableControls;
  end;
end;


procedure Register;
begin
  RegisterComponents('DBAddOns', [TPx7Table]);
end;

end.
